import SwiftUI
import Accelerate

/*
 Apply Additivity of Integrals
 
 Execute statements for this section at the end of listing. 
 
 This playground contains functions required for all playgrounds. 
 
 From: https://www.limit-point.com/blog/2023/tone-shaper/
 */

func userIFCurve(_ N:Int, points:[CGPoint]) -> [CGPoint] {
    
    guard let firstPoint = points.first, let lastPoint = points.last else {
        return []
    }
    
    let minX = firstPoint.x
    let maxX = lastPoint.x
    
    let scaleFactor = CGFloat(N) / (maxX - minX)
    
        // Map the points to integers in the [0, N] interval
    let scaledPoints:[CGPoint] = points.map { point in
        let scaledX = Int((point.x - minX) * scaleFactor)
        return CGPoint(x: CGFloat(scaledX), y: point.y)
    }
    
    return scaledPoints
}

func scale(sampleCount:Int, userIFCurve:[CGPoint]) -> Double {
    Double(sampleCount-1) / userIFCurve.last!.x
}

func allSampleIFValues_by_interpolateUserIFCurve(sampleCount:Int, userIFCurve:[CGPoint]) -> ([Double], [Double]) {
    
    let userIFIndices = userIFCurve.map { Double($0.x) }
    let userIFValues = userIFCurve.map { Double($0.y) }
    
    let scale = scale(sampleCount: sampleCount, userIFCurve: userIFCurve)
    
    let sampleIFIndices = userIFIndices.map { ($0 * scale).rounded() }
    
    return (vDSP.linearInterpolate(values: userIFValues, atIndices: sampleIFIndices), sampleIFIndices)
}

func interpolation_with_vDSP() {
    
    let userIFCurvePointCount = 500
    
    let points:[CGPoint] = [(0.0, 54.0), (0.231, 460.0), (0.846, 54.0), (1.0, 360.0)].map { CGPoint(x: CGFloat($0.0), y: CGFloat($0.1)) }
    
    let userIFCurve = userIFCurve(userIFCurvePointCount, points: points)
    
    let sampleIFValues = allSampleIFValues_by_interpolateUserIFCurve(sampleCount: 36, userIFCurve: userIFCurve).0
    
    let sampleIFValuesRounded = sampleIFValues.compactMap { value in
        Int(value.rounded())
    }
    
    print("points = \(points)\n")
    print("userIFCurve = \(userIFCurve)\n")
    print("sampleIFValuesRounded = \(sampleIFValuesRounded)\n")
}

func userIFValues_by_interpolateUserIFCurve(userIFCurve:[CGPoint]) -> [Double] {
    
    let indices = userIFCurve.map { Double($0.x) }
    let values = userIFCurve.map { Double($0.y) }
    
    return vDSP.linearInterpolate(values: values, atIndices: indices)
}

extension Double {
    func roundTo(_ places:Int) -> Double {
        let divisor = pow(10.0, Double(places))
        return (self * divisor).rounded() / divisor
    }
}

func createPartitionRanges(sampleCount: Int, rangeCount: Int) -> [ClosedRange<Int>] {
    
    guard rangeCount > 0 else {
        return []
    }
    
    var ranges: [ClosedRange<Int>] = []
    
    var start = 0
    while start < sampleCount {
        let end = min(start + rangeCount, sampleCount) - 1
        let range = start...end
        ranges.append(range)
        start = end + 1
    }
    
    return ranges
}

func sampleIFValuesForRange(scale:Double, sampleRange:ClosedRange<Int>, userIFValues:[Double]) -> [Double] {
        // scale sample range into user range
    let ta = Double(sampleRange.lowerBound) / scale 
    let tb = Double(sampleRange.upperBound) / scale 
    
    var valuesToInterpolate:[Double] = []
    var sampleIndices:[Double] = []
    
    func appendInterploatedValue(_ t:Double) {
        let delta = (t - t.rounded(.down))
        let index = Int(t.rounded(.down))
        if delta == 0 || (index+1 > userIFValues.count-1) { // index+1 may be out of bounds when delta = 0, or very nearly 0
            valuesToInterpolate.append(userIFValues[index])
        }
        else {
            let interpolated = userIFValues[index] * (1 - delta) + userIFValues[index+1] * delta
            valuesToInterpolate.append(interpolated)
        }
    }
    
    if ta == tb {
        appendInterploatedValue(ta) 
        sampleIndices.append(Double(sampleRange.lowerBound))
    }
    else {
        
            // start
        appendInterploatedValue(ta) 
        sampleIndices.append(Double(sampleRange.lowerBound))
        
            // middle, if any
        var lowerBound = Int(ta.rounded(.up))
        if lowerBound == Int(ta.rounded(.down)) {
            lowerBound += 1
        }
        var upperBound = Int(tb.rounded(.down))
        if upperBound == Int(tb.rounded(.up)) {
            upperBound -= 1
        }
        
        if lowerBound <= upperBound {
            valuesToInterpolate.append(contentsOf: Array(userIFValues[lowerBound...upperBound]))
            sampleIndices.append(contentsOf: (lowerBound...upperBound).map { Double($0) * scale })
        }
        
            // end
        appendInterploatedValue(tb) 
        sampleIndices.append(Double(sampleRange.upperBound))
        
    }
    
    sampleIndices = sampleIndices.map { $0 - sampleIndices[0]}
    
    return vDSP.linearInterpolate(values: valuesToInterpolate, atIndices: sampleIndices)
}

func integrate(samples:[Double], stepSize:Double, useSimpsons:Bool) -> [Double] {
    
    let sampleCount = samples.count
    var step = stepSize
    
    var result = [Double](repeating: 0.0, count: sampleCount)
    
    if useSimpsons {
        vDSP_vsimpsD(samples, 1, &step, &result, 1, vDSP_Length(sampleCount))
    }
    else {
        vDSP_vtrapzD(samples, 1, &step, &result, 1, vDSP_Length(sampleCount))
    }
    
    return result
}

func generateRandomArray(count: Int, inRange range: ClosedRange<Double>) -> [Double] {
    return (1...count).map { _ in Double.random(in: range) }
}

func piecewise_integration_random_array(sampleRate: Int = 44100, duration:Double = 1, bufferSize: Int = 512, useSimpsons:Bool) -> Double {
    
    let delta = 1.0 / Double(sampleRate) 
    
    let sampleCount = Int(duration * Double(sampleRate))
    
    let v = generateRandomArray(count: sampleCount, inRange: 20.0...22050)
    
    let partitions = createPartitionRanges(sampleCount: v.count, rangeCount: bufferSize)
    
    let integralWhole = integrate(samples: v, stepSize: delta, useSimpsons: useSimpsons)
    
    var combinedIntegral:[Double] = []
    
    var currentPartition = partitions[0]
    var partialV = Array(v[currentPartition])
    var lastIntegral = integrate(samples: partialV, stepSize: delta, useSimpsons: useSimpsons)
    
    combinedIntegral = combinedIntegral + lastIntegral
    
    if partitions.count > 1 {
        for i in 1...partitions.count-1 {
            currentPartition = partitions[i]
            partialV = [partialV.last!] + Array(v[currentPartition]) // add overlap
            let lastIntegralValue = lastIntegral.last!
            lastIntegral = integrate(samples: partialV, stepSize: delta, useSimpsons: useSimpsons)
            lastIntegral.removeFirst()
            lastIntegral = vDSP.add(lastIntegralValue, lastIntegral)
            combinedIntegral = combinedIntegral + lastIntegral
        }
    }
    
    var difference = vDSP.subtract(combinedIntegral, integralWhole)
    difference = vDSP.absolute(difference)
    
    let maxAbsDiff = vDSP.maximum(difference)
    
    print("max absolute difference = \(maxAbsDiff)")
    
    return maxAbsDiff
}

// Run - Apply Additivity of Integrals

let sampleRate =  4  // samples per unit interval [0,1)
let delta = 1.0 / Double(sampleRate) // 0.25

print(integrate(samples: [1,1,1,1,1,1,1,1,1,1,1], stepSize: delta, useSimpsons: false))
print(integrate(samples: [1,1,1,1,1,1,1], stepSize: 0.25, useSimpsons: false))
print(integrate(samples: [1,1,1,1,1] , stepSize: 0.25, useSimpsons: false))

var adjusted = vDSP.add(1.5, [0.0, 0.25, 0.5, 0.75, 1.0])
print(adjusted)

adjusted.removeFirst()
var full = [0.0, 0.25, 0.5, 0.75, 1.0, 1.25, 1.5] + adjusted
print(full)
